import 'dart:ui';

import 'package:asteroids/game/models/bullet.dart';
import 'package:asteroids/game/models/player.dart';
import 'package:asteroids/game/models/polygon_asteroid.dart';
import 'package:asteroids/game/services/game_service.dart';
import 'package:flutter_test/flutter_test.dart';

void main() {
  test('initializeAsteroids seeds asteroids and is idempotent', () {
    final service = GameService(initialPlayer: Player(x: 0, y: 0));

    service.initializeAsteroids(800, 600);
    final first = service.asteroids.first;

    service.initializeAsteroids(800, 600);

    expect(service.asteroidsInitialized, isTrue);
    expect(service.asteroids.length, GameService.asteroidCount);
    expect(identical(first, service.asteroids.first), isTrue);
  });

  test('shoot adds a bullet while playing', () {
    final service = GameService(initialPlayer: Player(x: 100, y: 100, angle: 0));

    service.shoot();

    expect(service.bullets.length, 1);
    final bullet = service.bullets.first;
    expect(bullet.vx, greaterThan(0));
    expect(bullet.vy, closeTo(0, 1e-9));
  });

  test('shoot does nothing when game is over', () {
    final service = GameService(initialPlayer: Player(x: 50, y: 50));

    service.initializeAsteroids(800, 600);
    service.asteroids
      ..clear()
      ..add(_asteroidAt(50, 50));

    service.updateAsteroids(0.0);
    expect(service.isGameOver, isTrue);

    service.shoot();

    expect(service.bullets, isEmpty);
  });

  test('updateAsteroids does nothing when game is over', () {
    final service = GameService(initialPlayer: Player(x: 50, y: 50, angle: 0));

    service.initializeAsteroids(800, 600);
    service.shoot();
    final bulletBefore = service.bullets.first.copyWith();

    service.asteroids
      ..clear()
      ..add(_asteroidAt(50, 50));

    service.updateAsteroids(0.0);
    expect(service.isGameOver, isTrue);

    service.updateAsteroids(1.0);

    expect(service.bullets.length, 1);
    final bulletAfter = service.bullets.first;
    expect(bulletAfter.x, bulletBefore.x);
    expect(bulletAfter.y, bulletBefore.y);
  });

  test('updateAsteroids moves asteroids when playing', () {
    final service = GameService(initialPlayer: Player(x: -10000, y: -10000));

    service.initializeAsteroids(800, 600);
    final first = service.asteroids.first;

    service.updateAsteroids(0.016);
    final updated = service.asteroids.first;

    final moved = first.x != updated.x || first.y != updated.y;
    expect(moved, isTrue);
  });

  test('bullets are removed when leaving the screen', () {
    final service = GameService(initialPlayer: Player(x: -10000, y: -10000));

    service.initializeAsteroids(100, 100);
    service.bullets.add(Bullet(x: 200, y: 200, vx: 0, vy: 0));

    service.updateAsteroids(0.016);

    expect(service.bullets, isEmpty);
  });

  test('bullet-asteroid collision removes both', () {
    final service = GameService(initialPlayer: Player(x: -10000, y: -10000));

    service.initializeAsteroids(800, 600);
    service.asteroids
      ..clear()
      ..add(_asteroidAt(50, 50));
    service.bullets
      ..clear()
      ..add(Bullet(x: 50, y: 50, vx: 0, vy: 0));

    service.updateAsteroids(0.016);

    expect(service.bullets, isEmpty);
    expect(service.asteroids, isEmpty);
  });

  test('reset clears bullets and restores playing state', () {
    final service = GameService(initialPlayer: Player(x: 100, y: 100));

    service.initializeAsteroids(800, 600);
    service.shoot();
    service.reset(Player(x: 50, y: 50, angle: 1.0), 800, 600);

    expect(service.isGameOver, isFalse);
    expect(service.bullets, isEmpty);
    expect(service.player.angle, 0.0);
    expect(service.asteroidsInitialized, isTrue);
  });
}

PolygonAsteroid _asteroidAt(double x, double y) {
  return PolygonAsteroid(
    x: x,
    y: y,
    radius: 12,
    vx: 0,
    vy: 0,
    vertices: const [
      Offset(0, -10),
      Offset(10, 10),
      Offset(-10, 10),
    ],
  );
}
