package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"os/exec"
	"strings"
)

var (
	classesQuery    string = `{"query":"query GetWorkspacesAndClasses {\n  allWorkspaceUsers {\n\t\tid\n    teacher {\n      id\n      email\n    }\n    workspace {\n      name\n      allClasses {\n        id\n        name\n      }\n    }\n  }\n}","variables":{},"operationName":"GetWorkspacesAndClasses"}`
	worksheetsQuery string = `{"query":"query GetWorksheetsForClass($classId: ID!) {\n  class(id: $classId) {\n    allWorksheetPlans {\n      id\n      name\n      topic {\n        id\n        emoji\n        title\n        color\n        __typename\n      }\n      subtopic {\n        id\n        icon\n        title\n        __typename\n      }\n      subtopics {\n        id\n        title\n        __typename\n      }\n      __typename\n    }\n    __typename\n  }\n  __typename\n}\n","variables":{"classId":".classID"},"operationName":"GetWorksheetsForClass"}`
)

var (
	env   string = "prod" // or "dev"
	DEBUG bool   = false  // Set to true to only process first email and first class
)

type FirebaseUser struct {
	Email  string `json:"email"`
	UserID string `json:"userId"`
	Found  bool   `json:"found"`
}

type RegionConfig struct {
	name string
	url  string
}

func main() {
	// Read emails from file
	emailFile := "template-emails.txt"
	if len(os.Args) > 1 {
		emailFile = os.Args[1]
	}

	emailBytes, err := os.ReadFile(emailFile)
	if err != nil {
		log.Fatalf("Failed to read email file: %v", err)
	}

	emails := []string{}
	for _, line := range strings.Split(string(emailBytes), "\n") {
		line = strings.TrimSpace(line)
		if line != "" && !strings.HasPrefix(line, "#") {
			emails = append(emails, line)
		}
	}

	if len(emails) == 0 {
		log.Fatal("No emails found in file")
	}

	fmt.Printf("Found %d emails in %s\n", len(emails), emailFile)

	// Get all Firebase UIDs
	scriptPath := "../../../../teaching/scripts/get-firebase-userids.js"
	args := append([]string{scriptPath, env}, emails...)
	cmd := exec.Command("node", args...)

	output, err := cmd.Output()
	if err != nil {
		log.Fatalf("Failed to get Firebase UIDs: %v", err)
	}

	var fbUsers []FirebaseUser
	if err := json.Unmarshal(output, &fbUsers); err != nil {
		log.Fatalf("Failed to parse Firebase response: %v", err)
	}

	// Filter only found users
	fmt.Println("\n=== Firebase UID Extraction ===")
	foundUsers := []FirebaseUser{}
	for _, user := range fbUsers {
		if user.Found {
			foundUsers = append(foundUsers, user)
			fmt.Printf("✓ %s -> %s\n", user.Email, user.UserID)
			if DEBUG {
				break // Only process first user in DEBUG mode
			}
		} else {
			fmt.Printf("✗ %s -> NOT FOUND\n", user.Email)
		}
	}

	if len(foundUsers) == 0 {
		log.Fatal("No users found in Firebase")
	}

	fmt.Printf("\nFound %d/%d users in Firebase\n", len(foundUsers), len(emails))

	// Create output file
	file, err := os.Create("worksheets.csv")
	if err != nil {
		log.Fatal(err)
	}
	file.WriteString("region,userId,userEmail,resourceId,resourceTitle,topicId,topicTitle,topicIcon,subtopicIds,subtopicTitles,subtopicIcon,className\n")
	defer file.Close()

	// Define regions
	regions := []RegionConfig{
		{name: "au", url: "http://localhost:3004"},
		{name: "us", url: "http://localhost:3005"},
	}

	// Process each user
	fmt.Println("\n=== Processing Users ===")
	for _, fbUser := range foundUsers {
		fmt.Printf("\n[%s]\n", fbUser.Email)

		// Determine preferred region based on email
		primaryRegion, fallbackRegion := getRegionPreference(fbUser.Email, regions)

		// Try primary region first
		wsCount := processUserInRegion(fbUser, primaryRegion, file)

		// If failed, try fallback region
		if wsCount == 0 {
			fmt.Printf("  No data in %s, trying %s region...\n", primaryRegion.name, fallbackRegion.name)
			wsCount = processUserInRegion(fbUser, fallbackRegion, file)
		}

		fmt.Printf("  Total worksheets exported: %d\n", wsCount)
	}

	fmt.Println("\n✓ Export complete! Saved to worksheets.csv")
}

func getRegionPreference(email string, regions []RegionConfig) (RegionConfig, RegionConfig) {
	emailLower := strings.ToLower(email)

	// If 'us' in email, prefer US
	if strings.Contains(emailLower, "usa") || strings.Contains(emailLower, "us") {
		return regions[1], regions[0] // US first, AU fallback
	}

	// If 'au' in email, prefer AU
	if strings.Contains(emailLower, "au") {
		return regions[0], regions[1] // AU first, US fallback
	}

	// Default to AU first
	return regions[0], regions[1]
}

func processUserInRegion(fbUser FirebaseUser, region RegionConfig, file *os.File) int {
	req, err := http.NewRequest("POST", region.url, bytes.NewBufferString(classesQuery))
	if err != nil {
		log.Printf("Error creating request for %s: %v", fbUser.Email, err)
		return 0
	}
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("X-UID", fbUser.UserID)

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		log.Printf("Error fetching classes for %s in %s: %v", fbUser.Email, region.name, err)
		return 0
	}

	var format format
	err = json.NewDecoder(resp.Body).Decode(&format)
	resp.Body.Close()
	if err != nil {
		log.Printf("Error decoding classes for %s in %s: %v", fbUser.Email, region.name, err)
		return 0
	}

	var classIDs []string
	var classNames []string
	for _, user := range format.Data.WorkspaceUsers {
		for _, class := range user.Workspace.AllClasses {
			classIDs = append(classIDs, class.ID)
			classNames = append(classNames, class.Name)
		}
	}

	if len(classIDs) == 0 {
		fmt.Printf("  %s region: 0 classes\n", region.name)
		return 0
	}

	fmt.Printf("  %s region: %d classes\n", region.name, len(classIDs))

	// Fetch worksheets for each class
	totalWS := 0
	classesToProcess := classIDs
	if DEBUG && len(classIDs) > 0 {
		classesToProcess = classIDs[:1] // Only process first class in DEBUG mode
		fmt.Printf("  DEBUG MODE: Processing only first class\n")
	}
	for idx, classID := range classesToProcess {
		worksheetWithClassID := strings.Replace(worksheetsQuery, ".classID", classID, 1)
		req, err := http.NewRequest("POST", region.url, bytes.NewBufferString(worksheetWithClassID))
		if err != nil {
			log.Printf("Error creating worksheet request: %v", err)
			continue
		}
		req.Header.Set("Content-Type", "application/json")
		req.Header.Set("X-UID", fbUser.UserID)

		resp, err := http.DefaultClient.Do(req)
		if err != nil {
			log.Printf("Error fetching worksheets: %v", err)
			continue
		}

		var ws worksheetResp
		err = json.NewDecoder(resp.Body).Decode(&ws)
		resp.Body.Close()
		if err != nil {
			log.Printf("Error decoding worksheets: %v", err)
			continue
		}

		classWSCount := 0
		for _, worksheet := range ws.Data.Class.AllWorksheetPlans {
			subIDs := []string{}
			subTitles := []string{}
			for _, subtopic := range worksheet.Subtopics {
				subIDs = append(subIDs, subtopic.ID)
				subTitles = append(subTitles, subtopic.Title)
			}
			subtopicIDs := fmt.Sprintf("\"%s\"", strings.Join(subIDs, ","))
			subtopicTitles := fmt.Sprintf("\"%s\"", strings.Join(subTitles, ","))

			wsStr := fmt.Sprintf("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
				region.name,
				fbUser.UserID,
				fbUser.Email,
				worksheet.ID,
				worksheet.Name,
				worksheet.Topic.ID,
				worksheet.Topic.Title,
				worksheet.Topic.Emoji,
				subtopicIDs,
				subtopicTitles,
				worksheet.Subtopic.Icon,
				classNames[idx])
			file.WriteString(wsStr)
			classWSCount++
			totalWS++
		}

		fmt.Printf("    Class %d/%d [%s]: %d worksheets\n", idx+1, len(classIDs), classNames[idx], classWSCount)
	}

	return totalWS
}

type workspaceUser struct {
	ID        string `json:"id"`
	Workspace struct {
		AllClasses []struct {
			ID   string `json:"id"`
			Name string `json:"name"`
		} `json:"allClasses"`
	} `json:"workspace"`
}

type format struct {
	Data struct {
		WorkspaceUsers []workspaceUser `json:"allWorkspaceUsers"`
	} `json:"data"`
}

type worksheetResp struct {
	Data struct {
		Class struct {
			AllWorksheetPlans []struct {
				ID    string `json:"id"`
				Name  string `json:"name"`
				Topic struct {
					ID    string `json:"id"`
					Emoji string `json:"emoji"`
					Title string `json:"title"`
					Color string `json:"color"`
				} `json:"topic"`
				Subtopic struct {
					ID    string `json:"id"`
					Icon  string `json:"icon"`
					Title string `json:"title"`
				} `json:"subtopic"`
				Subtopics []struct {
					ID    string `json:"id"`
					Title string `json:"title"`
				} `json:"subtopics"`
			} `json:"allWorksheetPlans"`
		} `json:"class"`
	} `json:"data"`
}
