package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"regexp"
	"strings"

	"cloud.google.com/go/firestore"
	firebase "firebase.google.com/go"
	"google.golang.org/api/iterator"
	"google.golang.org/api/option"
)

type SimulationInfo struct {
	Type    string `json:"type"`
	URL     string `json:"url,omitempty"`
	Details string `json:"details,omitempty"`
}

type TemplateAnalysis struct {
	ID          string           `json:"id"`
	HasSlides   bool             `json:"hasSlides"`
	SlideCount  int              `json:"slideCount"`
	Simulations []SimulationInfo `json:"simulations,omitempty"`
	URLs        []string         `json:"urls,omitempty"`
}

var knownNonSimulationDomains = []string{
	"youtube.com",
	"youtu.be",
	"vimeo.com",
	"google.com",
	"googleapis.com",
	"firebasestorage.googleapis.com",
	"gstatic.com",
	"cloudinary.com",
	"imgur.com",
	"wikipedia.org",
	"wikimedia.org",
}

var geogebraPattern = regexp.MustCompile(`(?i)geogebra\.org|geogebra\.com|ggb`)
var urlPattern = regexp.MustCompile(`https?://[^\s"'<>\)\]\\]+`)
var iframePattern = regexp.MustCompile(`(?i)<iframe[^>]*src=["']([^"']+)["']`)

func main() {
	if len(os.Args) < 3 {
		fmt.Println("Usage: inspect-templates <credentials-file> <project-id> [template-id|--sample N|--all]")
		fmt.Println("")
		fmt.Println("Examples:")
		fmt.Println("  inspect-templates creds.json tutero-dev template123")
		fmt.Println("  inspect-templates creds.json tutero-dev --sample 10")
		fmt.Println("  inspect-templates creds.json tutero-dev --all")
		os.Exit(1)
	}

	credsFile := os.Args[1]
	projectID := os.Args[2]
	mode := "--sample"
	if len(os.Args) > 3 {
		mode = os.Args[3]
	}

	ctx := context.Background()
	opt := option.WithCredentialsFile(credsFile)
	conf := &firebase.Config{ProjectID: projectID}

	app, err := firebase.NewApp(ctx, conf, opt)
	if err != nil {
		log.Fatalf("Firebase init failed: %v", err)
	}

	client, err := app.Firestore(ctx)
	if err != nil {
		log.Fatalf("Firestore client failed: %v", err)
	}
	defer client.Close()

	collection := "whiteboard-templates"

	switch {
	case mode == "--sample":
		sampleCount := 20
		if len(os.Args) > 4 {
			fmt.Sscanf(os.Args[4], "%d", &sampleCount)
		}
		fmt.Printf("Sampling %d templates from %s...\n\n", sampleCount, collection)
		sampleTemplates(ctx, client, collection, sampleCount)

	case mode == "--all":
		fmt.Printf("Analyzing all templates in %s (this may take a while)...\n\n", collection)
		analyzeAllTemplates(ctx, client, collection)

	case mode == "--dump":
		if len(os.Args) < 5 {
			log.Fatal("Usage: --dump <template-id>")
		}
		templateID := os.Args[4]
		dumpTemplate(ctx, client, collection, templateID)

	default:
		templateID := mode
		fmt.Printf("Inspecting template: %s\n\n", templateID)
		inspectTemplate(ctx, client, collection, templateID)
	}
}

func sampleTemplates(ctx context.Context, client *firestore.Client, collection string, count int) {
	iter := client.Collection(collection).Limit(count).Documents(ctx)
	
	var withSimulations []TemplateAnalysis
	var allURLs []string
	
	i := 0
	for {
		doc, err := iter.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			log.Printf("Error reading doc: %v", err)
			continue
		}

		i++
		analysis := analyzeTemplateData(doc.Ref.ID, doc.Data())
		
		fmt.Printf("[%d] %s - Slides: %d, Simulations: %d\n", 
			i, analysis.ID, analysis.SlideCount, len(analysis.Simulations))
		
		if len(analysis.Simulations) > 0 {
			withSimulations = append(withSimulations, analysis)
			for _, sim := range analysis.Simulations {
				fmt.Printf("    -> %s: %s\n", sim.Type, sim.URL)
			}
		}
		
		allURLs = append(allURLs, analysis.URLs...)
	}

	fmt.Printf("\n=== SUMMARY ===\n")
	fmt.Printf("Templates sampled: %d\n", i)
	fmt.Printf("With simulations: %d\n", len(withSimulations))
	
	fmt.Printf("\n=== UNIQUE URLs FOUND ===\n")
	uniqueURLs := dedupeStrings(allURLs)
	for _, url := range uniqueURLs {
		if !isKnownNonSimulation(url) {
			fmt.Printf("  %s\n", url)
		}
	}
}

func analyzeAllTemplates(ctx context.Context, client *firestore.Client, collection string) {
	iter := client.Collection(collection).Documents(ctx)
	
	simulationCounts := make(map[string]int)
	var templatesWithSims []string
	total := 0
	
	for {
		doc, err := iter.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			log.Printf("Error reading doc: %v", err)
			continue
		}

		total++
		if total%100 == 0 {
			fmt.Printf("\rProcessed: %d", total)
		}

		analysis := analyzeTemplateData(doc.Ref.ID, doc.Data())
		
		if len(analysis.Simulations) > 0 {
			templatesWithSims = append(templatesWithSims, analysis.ID)
			for _, sim := range analysis.Simulations {
				simulationCounts[sim.Type]++
			}
		}
	}

	fmt.Printf("\n\n=== FINAL SUMMARY ===\n")
	fmt.Printf("Total templates: %d\n", total)
	fmt.Printf("With simulations: %d\n", len(templatesWithSims))
	
	fmt.Printf("\nSimulation types:\n")
	for simType, count := range simulationCounts {
		fmt.Printf("  %s: %d\n", simType, count)
	}
	
	// Write templates with simulations to file
	data, _ := json.MarshalIndent(templatesWithSims, "", "  ")
	os.WriteFile("templates-with-simulations.json", data, 0644)
	fmt.Println("\nWrote templates-with-simulations.json")
}

func inspectTemplate(ctx context.Context, client *firestore.Client, collection string, templateID string) {
	docRef := client.Collection(collection).Doc(templateID)
	doc, err := docRef.Get(ctx)
	if err != nil {
		log.Fatalf("Failed to get template: %v", err)
	}
	
	if !doc.Exists() {
		fmt.Println("Template not found")
		return
	}

	data := doc.Data()
	analysis := analyzeTemplateData(templateID, data)
	
	fmt.Printf("Template ID: %s\n", analysis.ID)
	fmt.Printf("Has Slides: %v\n", analysis.HasSlides)
	fmt.Printf("Slide Count: %d\n", analysis.SlideCount)
	
	if len(analysis.Simulations) > 0 {
		fmt.Printf("\n=== SIMULATIONS DETECTED ===\n")
		for _, sim := range analysis.Simulations {
			fmt.Printf("  Type: %s\n", sim.Type)
			if sim.URL != "" {
				fmt.Printf("  URL: %s\n", sim.URL)
			}
			if sim.Details != "" {
				fmt.Printf("  Details: %s\n", sim.Details)
			}
			fmt.Println()
		}
	}
	
	if len(analysis.URLs) > 0 {
		fmt.Printf("\n=== ALL URLs ===\n")
		for _, url := range analysis.URLs {
			marker := ""
			if !isKnownNonSimulation(url) {
				marker = " [POTENTIAL SIMULATION]"
			}
			fmt.Printf("  %s%s\n", url, marker)
		}
	}
}

func dumpTemplate(ctx context.Context, client *firestore.Client, collection string, templateID string) {
	docRef := client.Collection(collection).Doc(templateID)
	doc, err := docRef.Get(ctx)
	if err != nil {
		log.Fatalf("Failed to get template: %v", err)
	}
	
	if !doc.Exists() {
		fmt.Println("Template not found")
		return
	}

	data := doc.Data()
	jsonBytes, _ := json.MarshalIndent(data, "", "  ")
	fmt.Println(string(jsonBytes))
}

func analyzeTemplateData(id string, data map[string]interface{}) TemplateAnalysis {
	analysis := TemplateAnalysis{
		ID: id,
	}

	jsonBytes, _ := json.Marshal(data)
	jsonStr := string(jsonBytes)

	if slides, ok := data["slides"].([]interface{}); ok {
		analysis.HasSlides = true
		analysis.SlideCount = len(slides)
	}

	urls := urlPattern.FindAllString(jsonStr, -1)
	analysis.URLs = dedupeStrings(urls)

	simulations := []SimulationInfo{}

	if geogebraPattern.MatchString(jsonStr) {
		geoURLs := geogebraPattern.FindAllString(jsonStr, -1)
		simulations = append(simulations, SimulationInfo{
			Type:    "geogebra",
			Details: strings.Join(dedupeStrings(geoURLs), ", "),
		})
	}

	iframeMatches := iframePattern.FindAllStringSubmatch(jsonStr, -1)
	for _, match := range iframeMatches {
		if len(match) > 1 {
			iframeSrc := match[1]
			if !isKnownNonSimulation(iframeSrc) {
				simulations = append(simulations, SimulationInfo{
					Type: "iframe",
					URL:  iframeSrc,
				})
			}
		}
	}

	for _, url := range analysis.URLs {
		if !isKnownNonSimulation(url) && !geogebraPattern.MatchString(url) {
			if isLikelySimulation(url) {
				simulations = append(simulations, SimulationInfo{
					Type: "external",
					URL:  url,
				})
			}
		}
	}

	analysis.Simulations = dedupeSimulations(simulations)
	return analysis
}

func isKnownNonSimulation(url string) bool {
	lowerURL := strings.ToLower(url)
	for _, domain := range knownNonSimulationDomains {
		if strings.Contains(lowerURL, domain) {
			return true
		}
	}
	if strings.HasSuffix(lowerURL, ".png") ||
		strings.HasSuffix(lowerURL, ".jpg") ||
		strings.HasSuffix(lowerURL, ".jpeg") ||
		strings.HasSuffix(lowerURL, ".gif") ||
		strings.HasSuffix(lowerURL, ".svg") ||
		strings.HasSuffix(lowerURL, ".webp") ||
		strings.HasSuffix(lowerURL, ".mp4") ||
		strings.HasSuffix(lowerURL, ".mp3") ||
		strings.HasSuffix(lowerURL, ".pdf") {
		return true
	}
	return false
}

func isLikelySimulation(url string) bool {
	lowerURL := strings.ToLower(url)
	
	simulationIndicators := []string{
		"phet.colorado.edu",
		"desmos.com",
		"mathigon.org",
		"nctm.org",
		"illuminations.nctm.org",
		"interactivate.com",
		"shodor.org",
		"explorelearning.com",
		"mathsisfun.com",
		"mathplayground.com",
		"visnos.com",
		"nrich.maths.org",
		"transum.org",
		"intmath.com",
		"mathopenref.com",
		"mathwarehouse.com",
		"khanacademy.org/computer-programming",
		"scratch.mit.edu",
		"codepen.io",
		"jsfiddle.net",
		"jsbin.com",
		"glitch.com",
		"polypad.org",
		"teacher.desmos.com",
		"brilliant.org",
		"wolframalpha.com",
		"/embed",
		"/app/",
		"simulation",
		"interactive",
		"applet",
	}

	for _, indicator := range simulationIndicators {
		if strings.Contains(lowerURL, indicator) {
			return true
		}
	}

	return false
}

func dedupeStrings(strs []string) []string {
	seen := make(map[string]bool)
	result := []string{}
	for _, s := range strs {
		if !seen[s] {
			seen[s] = true
			result = append(result, s)
		}
	}
	return result
}

func dedupeSimulations(sims []SimulationInfo) []SimulationInfo {
	seen := make(map[string]bool)
	result := []SimulationInfo{}
	for _, s := range sims {
		key := s.Type + ":" + s.URL
		if !seen[key] {
			seen[key] = true
			result = append(result, s)
		}
	}
	return result
}
