package main

import (
	"bufio"
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"strings"
	"sync"

	firebase "firebase.google.com/go"
	"google.golang.org/api/option"
)

func main() {
	if len(os.Args) < 4 {
		log.Fatal("Usage: firebase-check <credentials-file> <project-id> <ids-file>")
	}

	credsFile := os.Args[1]
	projectID := os.Args[2]
	idsFile := os.Args[3]

	ctx := context.Background()
	opt := option.WithCredentialsFile(credsFile)
	conf := &firebase.Config{ProjectID: projectID}

	app, err := firebase.NewApp(ctx, conf, opt)
	if err != nil {
		log.Fatalf("Firebase init failed: %v", err)
	}

	client, err := app.Firestore(ctx)
	if err != nil {
		log.Fatalf("Firestore client failed: %v", err)
	}
	defer client.Close()

	ids, err := readIDs(idsFile)
	if err != nil {
		log.Fatalf("Failed to read IDs: %v", err)
	}

	fmt.Printf("Checking %d templates (50 workers)...\n", len(ids))

	type result struct {
		id     string
		exists bool
	}

	jobs := make(chan string, len(ids))
	results := make(chan result, len(ids))

	var wg sync.WaitGroup
	for i := 0; i < 50; i++ {
		wg.Add(1)
		go func() {
			defer wg.Done()
			for id := range jobs {
				docRef := client.Collection("whiteboard-templates").Doc(id)
				docSnap, err := docRef.Get(ctx)
				exists := err == nil && docSnap.Exists()
				results <- result{id, exists}
			}
		}()
	}

	for _, id := range ids {
		jobs <- id
	}
	close(jobs)

	go func() {
		wg.Wait()
		close(results)
	}()

	var missing []string
	checked := 0
	for r := range results {
		checked++
		if checked%100 == 0 {
			fmt.Printf("\r[%d/%d]", checked, len(ids))
		}
		if !r.exists {
			missing = append(missing, r.id)
		}
	}
	fmt.Println()

	fmt.Printf("Found: %d | Missing: %d\n", len(ids)-len(missing), len(missing))

	data, _ := json.MarshalIndent(missing, "", "  ")
	if err := os.WriteFile("new-list.json", data, 0644); err != nil {
		log.Fatalf("Failed to write new-list.json: %v", err)
	}
}

func readIDs(filename string) ([]string, error) {
	file, err := os.Open(filename)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	var ids []string
	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		id := strings.TrimSpace(scanner.Text())
		if id != "" && !strings.HasPrefix(id, "#") {
			ids = append(ids, id)
		}
	}
	return ids, scanner.Err()
}
