package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"strings"

	firebase "firebase.google.com/go"
	"google.golang.org/api/iterator"
	"google.golang.org/api/option"
)

func main() {
	if len(os.Args) < 4 {
		fmt.Println("Usage: find-simulations <credentials-file> <project-id> <output-file>")
		fmt.Println("")
		fmt.Println("Scans whiteboard-templates collection for simulations")
		fmt.Println("(templates with type:url + urlType:iframe)")
		os.Exit(1)
	}

	credsFile := os.Args[1]
	projectID := os.Args[2]
	outputFile := os.Args[3]

	ctx := context.Background()
	opt := option.WithCredentialsFile(credsFile)
	conf := &firebase.Config{ProjectID: projectID}

	app, err := firebase.NewApp(ctx, conf, opt)
	if err != nil {
		log.Fatalf("Firebase init failed: %v", err)
	}

	client, err := app.Firestore(ctx)
	if err != nil {
		log.Fatalf("Firestore client failed: %v", err)
	}
	defer client.Close()

	collection := "whiteboard-templates"

	fmt.Printf("Scanning %s for simulations...\n", collection)

	iter := client.Collection(collection).Documents(ctx)

	var simulationIDs []string
	scanned := 0

	for {
		doc, err := iter.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			log.Printf("Error reading doc: %v", err)
			continue
		}

		scanned++
		if scanned%1000 == 0 {
			fmt.Printf("\r  Scanned: %d", scanned)
		}

		data := doc.Data()
		jsonBytes, _ := json.Marshal(data)
		jsonStr := string(jsonBytes)

		// Check for simulation pattern: type:"url" + urlType:"iframe"
		hasURLType := strings.Contains(jsonStr, `"type":"url"`) || strings.Contains(jsonStr, `"type": "url"`)
		hasIframe := strings.Contains(jsonStr, `"urlType":"iframe"`) || strings.Contains(jsonStr, `"urlType": "iframe"`)

		if hasURLType && hasIframe {
			simulationIDs = append(simulationIDs, doc.Ref.ID)
		}
	}

	fmt.Printf("\r  Scanned: %d\n", scanned)
	fmt.Printf("  Found %d simulation templates\n", len(simulationIDs))

	// Write to output file
	output, err := json.MarshalIndent(simulationIDs, "", "  ")
	if err != nil {
		log.Fatalf("Failed to marshal JSON: %v", err)
	}

	if err := os.WriteFile(outputFile, output, 0644); err != nil {
		log.Fatalf("Failed to write output file: %v", err)
	}

	fmt.Printf("  Wrote %s\n", outputFile)
}
