#!/usr/bin/env bash
set -euo pipefail

###############################################################################
# Find Simulation Templates Script
# 
# 1. Scans Firebase whiteboard-templates for simulations (type:url + urlType:iframe)
# 2. Outputs template IDs to JSON
# 3. Queries Neo4j to find linked skills
# 4. Outputs CSV: skillId, templateId, templatePosition
#
# Usage: ./find-simulation-templates.sh [dev|prod]
###############################################################################

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

# Environment selection
ENV="${1:-dev}"

case "$ENV" in
  dev)
    FIREBASE_PROJECT="mathgaps-dev-b044f"
    FIREBASE_CREDS="../../secrets/sa-dev.json"
    CYPHER_CMD="cypher-safe --preset learning-dev"
    ;;
  prod)
    FIREBASE_PROJECT="mathgaps-56d5a"
    FIREBASE_CREDS="../../secrets/sa-prod.json"
    # Requires port-forward to 7680
    export NEO4J_PASSWORD="De0YFd4XG239RCoP"
    CYPHER_CMD="cypher-safe -h localhost -p 7680"
    ;;
  *)
    echo "Usage: $0 [dev|prod]"
    exit 1
    ;;
esac

OUTPUT_DIR="$SCRIPT_DIR/output-simulations-$ENV"
mkdir -p "$OUTPUT_DIR"

echo "Environment: $ENV"
echo "Firebase: $FIREBASE_PROJECT"
echo "Output: $OUTPUT_DIR"

log_step() {
  echo ""
  echo "========================================"
  echo "[$1] $2"
  echo "========================================"
}

log_info() {
  echo "  -> $1"
}

###############################################################################
# STEP 1: Find simulation templates in Firebase
###############################################################################
log_step "1" "Scanning Firebase for simulation templates"

go run find-simulations.go "$FIREBASE_CREDS" "$FIREBASE_PROJECT" "$OUTPUT_DIR/simulation-template-ids.json"

sim_count=$(jq length "$OUTPUT_DIR/simulation-template-ids.json")
log_info "Found $sim_count simulation templates"

if [[ "$sim_count" -eq 0 ]]; then
  log_info "No simulations found. Exiting."
  exit 0
fi

log_info "Sample IDs:"
jq -r '.[:5][]' "$OUTPUT_DIR/simulation-template-ids.json" | sed 's/^/      /'

###############################################################################
# STEP 2: Query Neo4j for skills linked to these templates
###############################################################################
log_step "2" "Querying Neo4j (learning-dev) for linked skills"

# Read template IDs into a JSON array for the query
TEMPLATE_IDS=$(cat "$OUTPUT_DIR/simulation-template-ids.json")

# Cypher query to find skills and template positions
read -r -d '' QUERY <<'CYPHER' || true
WITH $templateIds AS templateIds
UNWIND templateIds AS searchTemplateId
MATCH (skillSection:SkillSection)-[:HAS]->(skill:Skill)
MATCH (skillSection)-[:INCLUDES]->(slide:LessonPlanSkillSlide)
WHERE searchTemplateId IN slide.skillTemplateIDs
WITH skill, slide, searchTemplateId,
    CASE
        WHEN slide.templateOrder IS NOT NULL AND size(slide.templateOrder) > 0
        THEN slide.templateOrder
        ELSE slide.skillTemplateIDs
    END AS orderList
UNWIND range(0, size(orderList) - 1) AS idx
WITH skill, searchTemplateId, idx, orderList[idx] AS templateAtIdx
WHERE templateAtIdx = searchTemplateId
RETURN DISTINCT skill.id AS skillId, searchTemplateId AS templateId, (idx + 1) AS templatePosition
ORDER BY skillId, templatePosition
CYPHER

log_info "Running query..."
PARAMS="{\"templateIds\": $TEMPLATE_IDS}"

$CYPHER_CMD --params "$PARAMS" "$QUERY" > "$OUTPUT_DIR/skills-with-simulations.json"

result_count=$(jq length "$OUTPUT_DIR/skills-with-simulations.json")
log_info "Found $result_count skill-template associations"

###############################################################################
# STEP 3: Generate CSV output
###############################################################################
log_step "3" "Generating CSV output"

# Convert JSON to CSV
echo "skillId,templateId,templatePosition" > "$OUTPUT_DIR/simulation-skills.csv"
jq -r '.[] | [.skillId, .templateId, .templatePosition] | @csv' "$OUTPUT_DIR/skills-with-simulations.json" >> "$OUTPUT_DIR/simulation-skills.csv"

csv_lines=$(($(wc -l < "$OUTPUT_DIR/simulation-skills.csv") - 1))
log_info "Wrote $csv_lines rows to simulation-skills.csv"

###############################################################################
# Summary
###############################################################################
log_step "DONE" "Complete"

echo ""
echo "  Output files in $OUTPUT_DIR/:"
echo "    simulation-template-ids.json  - Template IDs with simulations ($sim_count)"
echo "    skills-with-simulations.json  - Neo4j query results ($result_count)"
echo "    simulation-skills.csv         - Final CSV ($csv_lines rows)"
echo ""
echo "  CSV columns: skillId, templateId, templatePosition"
echo ""

# Show sample output
log_info "Sample CSV rows:"
head -6 "$OUTPUT_DIR/simulation-skills.csv" | sed 's/^/      /'
