package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"regexp"
	"strings"
	firebase "firebase.google.com/go"
	"google.golang.org/api/iterator"
	"google.golang.org/api/option"
)

func main() {
	if len(os.Args) < 3 {
		fmt.Println("Usage: explore-templates <credentials-file> <project-id> [template-id]")
		os.Exit(1)
	}

	credsFile := os.Args[1]
	projectID := os.Args[2]

	ctx := context.Background()
	opt := option.WithCredentialsFile(credsFile)
	conf := &firebase.Config{ProjectID: projectID}

	app, err := firebase.NewApp(ctx, conf, opt)
	if err != nil {
		log.Fatalf("Firebase init failed: %v", err)
	}

	client, err := app.Firestore(ctx)
	if err != nil {
		log.Fatalf("Firestore client failed: %v", err)
	}
	defer client.Close()

	collection := "whiteboard-templates"

	if len(os.Args) > 3 {
		// Dump specific template
		templateID := os.Args[3]
		doc, err := client.Collection(collection).Doc(templateID).Get(ctx)
		if err != nil {
			log.Fatalf("Failed to get template: %v", err)
		}
		data := doc.Data()
		jsonBytes, _ := json.MarshalIndent(data, "", "  ")
		fmt.Println(string(jsonBytes))
		return
	}

	// Scan all templates
	iter := client.Collection(collection).Documents(ctx)
	
	i := 0
	urlPattern := regexp.MustCompile(`https?://[^\s"'<>\)\]\\]+`)
	
	for {
		doc, err := iter.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			log.Printf("Error: %v", err)
			continue
		}
		i++
		data := doc.Data()
		jsonBytes, _ := json.Marshal(data)
		jsonStr := string(jsonBytes)
		
		// Find all URLs
		urls := urlPattern.FindAllString(jsonStr, -1)
		
		// Check for URL type objects (actual embedded content)
		hasURLType := strings.Contains(jsonStr, `"type":"url"`) || strings.Contains(jsonStr, `"type": "url"`)
		hasIframe := strings.Contains(jsonStr, `"urlType":"iframe"`) || strings.Contains(jsonStr, `"urlType": "iframe"`)
		
		// Look for any "type":"url" to see all embed types
		if hasURLType {
			fmt.Printf("\n=== %s ===\n", doc.Ref.ID)
			fmt.Printf("hasIframe: %v\n", hasIframe)
			// Show urlType value
			if idx := strings.Index(jsonStr, `"urlType"`); idx > 0 {
				fmt.Printf("urlType context: %s\n", jsonStr[idx:min(idx+50, len(jsonStr))])
			}
			for _, u := range urls {
				if !strings.Contains(u, "storage.googleapis.com") && !strings.Contains(u, ".png") && !strings.Contains(u, ".jpg") {
					fmt.Printf("  %s\n", u)
				}
			}
		}
	}

	fmt.Printf("\nTotal templates scanned: %d\n", i)
}
