#!/usr/bin/env bash
set -euo pipefail

###############################################################################
#                              CONFIGURATION
###############################################################################

# Neo4j password for prod environments (dev uses preset)
NEO4J_PASSWORD="De0YFd4XG239RCoP"

# Redis port (same for all environments - you port-forward manually)
REDIS_PORT=6370

# Firebase projects
FIREBASE_DEV_PROJECT="mathgaps-dev-b044f"
FIREBASE_PROD_PROJECT="mathgaps-56d5a"

# Credentials paths (relative to script directory)
FIREBASE_DEV_CREDS="../../secrets/sa-app.json"
FIREBASE_PROD_CREDS="../../secrets/sa-prod.json"

# Batch size for cache key queries (to avoid Neo4j timeout)
BATCH_SIZE=100

# Session name for cache key queries (keeps session alive for debugging)
CACHE_SESSION="skill-template-cache-keys"

###############################################################################
#                              CYPHER QUERIES
###############################################################################

# Query: Get all unique skill template IDs from enabled lesson plan slides
read -r -d '' QUERY_UNIQUE_TEMPLATES <<'CYPHER' || true
MATCH (lpn:LessonPlanSkillSlide {enabled: true})
WHERE NOT lpn.skillTemplateIDs IS NULL
UNWIND lpn.skillTemplateIDs AS templateID
RETURN DISTINCT templateID
CYPHER

# Query: Get cache keys for lesson sections that use specific templates
# Note: Uses $templateIDs parameter
read -r -d '' QUERY_CACHE_KEYS <<'CYPHER' || true
UNWIND $templateIDs AS templateID
MATCH (lpn:LessonPlanSkillSlide {enabled: true})--(ls:LessonSection)
WHERE NOT lpn.skillTemplateIDs IS NULL 
  AND templateID IN lpn.skillTemplateIDs
SET lpn.enabled = false
RETURN DISTINCT "learning/section-nodes:" + ls.id AS cacheKey
CYPHER

###############################################################################
#                           SCRIPT STARTS HERE
###############################################################################

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

show_usage() {
  echo "Usage: $0 <environment>"
  echo ""
  echo "Environments:"
  echo "  dev      - Uses resources-dev preset, dev Firebase"
  echo "  prod     - Uses local Neo4j (7687), prod Firebase"
  echo "  prod-us  - Uses local Neo4j (7687), prod Firebase"
  echo ""
  echo "Prerequisites:"
  echo "  dev:           resources-dev preset configured"
  echo "  prod/prod-us:  port-forward Neo4j to localhost:7687"
  echo "  all:           Redis available on localhost:$REDIS_PORT"
}

log_step() {
  echo ""
  echo "========================================"
  echo "[$1] $2"
  echo "========================================"
}

log_info() {
  echo "  -> $1"
}

run_cypher_dev() {
  local query="$1"
  local params="${2:-}"
  local session="${3:-}"

  if [[ -n "$session" ]] && [[ -n "$params" ]]; then
    cypher-safe --preset resources-dev --session "$session" --params "$params" "$query"
  elif [[ -n "$session" ]]; then
    cypher-safe --preset resources-dev --session "$session" "$query"
  elif [[ -n "$params" ]]; then
    cypher-safe --preset resources-dev --params "$params" "$query"
  else
    cypher-safe --preset resources-dev "$query"
  fi
}

run_cypher_prod() {
  local query="$1"
  local params="${2:-}"
  local session="${3:-}"

  if [[ -n "$session" ]] && [[ -n "$params" ]]; then
    NEO4J_PASSWORD="$NEO4J_PASSWORD" cypher-safe --port 7687 --session "$session" --params "$params" "$query"
  elif [[ -n "$session" ]]; then
    NEO4J_PASSWORD="$NEO4J_PASSWORD" cypher-safe --port 7687 --session "$session" "$query"
  elif [[ -n "$params" ]]; then
    NEO4J_PASSWORD="$NEO4J_PASSWORD" cypher-safe --port 7687 --params "$params" "$query"
  else
    NEO4J_PASSWORD="$NEO4J_PASSWORD" cypher-safe --port 7687 "$query"
  fi
}

# Parse arguments
ENV="${1:-}"
if [[ -z "$ENV" ]]; then
  show_usage
  exit 1
fi

# Set environment-specific config
case "$ENV" in
dev)
  run_cypher() { run_cypher_dev "$@"; }
  FIREBASE_PROJECT="$FIREBASE_DEV_PROJECT"
  FIREBASE_CREDS="$FIREBASE_DEV_CREDS"
  ;;
prod | prod-us)
  run_cypher() { run_cypher_prod "$@"; }
  FIREBASE_PROJECT="$FIREBASE_PROD_PROJECT"
  FIREBASE_CREDS="$FIREBASE_PROD_CREDS"
  ;;
*)
  echo "Error: Unknown environment '$ENV'"
  show_usage
  exit 1
  ;;
esac

# Setup output directory
OUTPUT_DIR="$SCRIPT_DIR/output-$ENV"
mkdir -p "$OUTPUT_DIR"
cd "$SCRIPT_DIR"

echo ""
echo "Environment:  $ENV"
echo "Firebase:     $FIREBASE_PROJECT"
echo "Output:       $OUTPUT_DIR"

###############################################################################
# STEP 1: Fetch unique template IDs from Neo4j
###############################################################################
log_step "1" "Fetching unique template IDs from Neo4j"

run_cypher "$QUERY_UNIQUE_TEMPLATES" | jq '[.[].templateID]' >"$OUTPUT_DIR/unique-ids.json"

unique_count=$(jq length "$OUTPUT_DIR/unique-ids.json")
log_info "Found $unique_count unique template IDs"

# Create line-by-line version for Go program
jq -r '.[]' "$OUTPUT_DIR/unique-ids.json" >"$OUTPUT_DIR/sample-ids.txt"

###############################################################################
# STEP 2: Check templates against Firebase
###############################################################################
log_step "2" "Checking templates against Firebase"

(
  cd "$OUTPUT_DIR"
  go run "$SCRIPT_DIR/firebase-check.go" \
    "$SCRIPT_DIR/$FIREBASE_CREDS" \
    "$FIREBASE_PROJECT" \
    sample-ids.txt
)

raw_missing=$(jq length "$OUTPUT_DIR/new-list.json")
log_info "Missing from Firebase (raw): $raw_missing"

# Filter out overview_* templates (dynamically generated, not stored in Firebase)
mv "$OUTPUT_DIR/new-list.json" "$OUTPUT_DIR/new-list-raw.json"
jq '[.[] | select(startswith("overview_") | not)]' "$OUTPUT_DIR/new-list-raw.json" >"$OUTPUT_DIR/new-list.json"
missing_count=$(jq length "$OUTPUT_DIR/new-list.json")
log_info "After filtering overview_*: $missing_count"

###############################################################################
# STEP 3: Compare with past-list.json
###############################################################################
log_step "3" "Comparing with past-list.json"

if [[ -f "past-list.json" ]]; then
  # Templates in new-list but NOT in past-list = newly missing
  jq -s '.[0] - .[1]' "$OUTPUT_DIR/new-list.json" past-list.json >"$OUTPUT_DIR/newly-missing.json"

  past_count=$(jq length past-list.json)
  newly_count=$(jq length "$OUTPUT_DIR/newly-missing.json")

  log_info "Previously known: $past_count"
  log_info "Currently missing: $missing_count"
  log_info "Newly missing: $newly_count"
else
  log_info "No past-list.json found"
  cp "$OUTPUT_DIR/new-list.json" "$OUTPUT_DIR/newly-missing.json"
fi

###############################################################################
# STEP 4: Get cache keys (batched, using session to keep alive)
###############################################################################
log_step "4" "Fetching cache keys for missing templates"

total_missing=$(jq length "$OUTPUT_DIR/new-list.json")
num_batches=$(((total_missing + BATCH_SIZE - 1) / BATCH_SIZE))

# Clear cache keys file
: >"$OUTPUT_DIR/cachekeys.txt"

log_info "Using session: $CACHE_SESSION (will NOT be dropped)"

for ((i = 0; i < num_batches; i++)); do
  start=$((i * BATCH_SIZE))
  batch_ids=$(jq ".[$start:$((start + BATCH_SIZE))]" "$OUTPUT_DIR/new-list.json")

  log_info "Batch $((i + 1))/$num_batches"

  params="{\"templateIDs\": $batch_ids}"
  run_cypher "$QUERY_CACHE_KEYS" "$params" "$CACHE_SESSION" 2>/dev/null |
    jq -r '.[].cacheKey' >>"$OUTPUT_DIR/cachekeys.txt" || true
done

# Deduplicate
sort -u "$OUTPUT_DIR/cachekeys.txt" -o "$OUTPUT_DIR/cachekeys.txt"

cache_count=$(wc -l <"$OUTPUT_DIR/cachekeys.txt" | tr -d ' ')
log_info "Unique cache keys: $cache_count"
log_info "Session '$CACHE_SESSION' kept alive - use 'cypher-safe --show $CACHE_SESSION' to inspect"

###############################################################################
# STEP 5: Redis deletion
###############################################################################
log_step "5" "Redis cache invalidation"

if [[ "$cache_count" -eq 0 ]]; then
  log_info "No cache keys to delete"
else
  log_info "Keys to delete: $cache_count"
  log_info "Sample keys:"
  head -5 "$OUTPUT_DIR/cachekeys.txt" | sed 's/^/      /'

  echo ""
  echo "  To delete manually:"
  echo "    while read -r key; do redis-cli -p $REDIS_PORT DEL \"\$key\"; done < $OUTPUT_DIR/cachekeys.txt"
  echo ""

  read -rp "  Delete now? (y/N): " confirm
  if [[ "$confirm" == "y" || "$confirm" == "Y" ]]; then
    deleted=0
    while IFS= read -r key; do
      redis-cli -p "$REDIS_PORT" DEL "$key" >/dev/null
      ((deleted++)) || true
    done <"$OUTPUT_DIR/cachekeys.txt"
    log_info "Deleted $deleted keys"
  else
    log_info "Skipped"
  fi
fi

###############################################################################
# Summary
###############################################################################
log_step "DONE" "Complete"

echo ""
echo "  Output files in $OUTPUT_DIR/:"
echo "    unique-ids.json     - All templates from Neo4j ($unique_count)"
echo "    new-list.json       - Missing from Firebase ($missing_count)"
echo "    newly-missing.json  - New missing (not in past-list)"
echo "    cachekeys.txt       - Redis keys ($cache_count)"
echo ""
echo "  Session kept alive: $CACHE_SESSION"
echo "    View:  cypher-safe --show $CACHE_SESSION"
echo "    Drop:  cypher-safe --drop $CACHE_SESSION"
echo ""
