package main

import (
	"context"
	"fmt"
	"log"
	"os"
	"sync"
	"sync/atomic"

	firebase "firebase.google.com/go"
	"google.golang.org/api/iterator"
	"google.golang.org/api/option"
)

func main() {
	if len(os.Args) < 3 {
		fmt.Println("Usage: backup-templates <credentials-file> <project-id> [--dry-run|--backup]")
		fmt.Println("")
		fmt.Println("Modes:")
		fmt.Println("  --dry-run  Count documents only (default)")
		fmt.Println("  --backup   Copy to whiteboard-templates-backup collection")
		os.Exit(1)
	}

	credsFile := os.Args[1]
	projectID := os.Args[2]
	mode := "--dry-run"
	if len(os.Args) > 3 {
		mode = os.Args[3]
	}

	ctx := context.Background()
	opt := option.WithCredentialsFile(credsFile)
	conf := &firebase.Config{ProjectID: projectID}

	app, err := firebase.NewApp(ctx, conf, opt)
	if err != nil {
		log.Fatalf("Firebase init failed: %v", err)
	}

	client, err := app.Firestore(ctx)
	if err != nil {
		log.Fatalf("Firestore client failed: %v", err)
	}
	defer client.Close()

	srcCollection := "whiteboard-templates"
	dstCollection := "whiteboard-templates-backup"

	// Count documents first
	fmt.Printf("Counting documents in %s...\n", srcCollection)
	
	iter := client.Collection(srcCollection).Documents(ctx)
	var docs []struct {
		id   string
		data map[string]interface{}
	}
	
	for {
		doc, err := iter.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			log.Fatalf("Error iterating: %v", err)
		}
		docs = append(docs, struct {
			id   string
			data map[string]interface{}
		}{doc.Ref.ID, doc.Data()})
	}

	fmt.Printf("Found %d documents in %s\n", len(docs), srcCollection)

	if mode == "--dry-run" {
		fmt.Println("\nDry run complete. Use --backup to copy to backup collection.")
		return
	}

	// Backup mode
	fmt.Printf("\nBacking up to %s...\n", dstCollection)

	var copied int64
	var failed int64
	var wg sync.WaitGroup
	sem := make(chan struct{}, 50) // 50 concurrent writes

	for i, doc := range docs {
		wg.Add(1)
		sem <- struct{}{}
		
		go func(idx int, id string, data map[string]interface{}) {
			defer wg.Done()
			defer func() { <-sem }()

			_, err := client.Collection(dstCollection).Doc(id).Set(ctx, data)
			if err != nil {
				atomic.AddInt64(&failed, 1)
				fmt.Printf("\rFailed: %s - %v", id, err)
			} else {
				atomic.AddInt64(&copied, 1)
			}

			if c := atomic.LoadInt64(&copied); c%100 == 0 {
				fmt.Printf("\r[%d/%d] Copied...", c, len(docs))
			}
		}(i, doc.id, doc.data)
	}

	wg.Wait()
	fmt.Println()

	fmt.Printf("\n=== BACKUP COMPLETE ===\n")
	fmt.Printf("Total: %d | Copied: %d | Failed: %d\n", len(docs), copied, failed)
}
