import 'package:asteroids/model/particle_model.dart';
import 'package:flutter/material.dart';

class AsteroidsView extends StatelessWidget {
  final List<Particle> particles;
  final List<Bullet> bullets; // Add bullets list
  final Offset cursorPosition;
  final double cursorAngle; // Angle for cursor rotation
  final bool isStopped;
  final Duration elapsedTime;
  final Function(Offset) onMouseHover;
  final Function() onTryAgainPressed;
  final Function() onMouseClick; // Add onMouseClick callback

  const AsteroidsView({
    required this.particles,
    required this.bullets, // Add bullets parameter
    required this.cursorPosition,
    required this.cursorAngle, // Angle for cursor rotation
    required this.isStopped,
    required this.elapsedTime,
    required this.onMouseHover,
    required this.onTryAgainPressed,
    required this.onMouseClick,
    super.key,
  });

  @override
  Widget build(BuildContext context) {
    final timerText = _formatDuration(elapsedTime);
    final gameOverTime = _formatDuration(elapsedTime, isGameOver: true);

    return GestureDetector(
      onPanUpdate: (details) {
        onMouseHover(details.localPosition);
      },
      onTap: onMouseClick, // Call onMouseClick on mouse click
      child: Stack(
        children: [
          if (isStopped)
            Center(
              child: Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  const Text(
                    "GAME OVER!",
                    style: TextStyle(color: Colors.white, fontSize: 35),
                  ),
                  const SizedBox(height: 10),
                  Text(
                    gameOverTime,
                    style: const TextStyle(color: Colors.white, fontSize: 25),
                  ),
                  const SizedBox(height: 10),
                  ElevatedButton(
                    onPressed: onTryAgainPressed,
                    child: const Text("Try Again"),
                  ),
                ],
              ),
            )
          else
            MouseRegion(
              onHover: (event) {
                onMouseHover(event.localPosition);
              },
              child: Stack(
                children: [
                  ...particles.map((particle) => Positioned(
                        left: particle.position.dx,
                        top: particle.position.dy,
                        child: Container(
                          width: particle.size,
                          height: particle.size,
                          decoration: const BoxDecoration(
                            color: Colors.red,
                            shape: BoxShape.circle,
                          ),
                        ),
                      )),
                  ...bullets.map((bullet) => Positioned(
                        left: bullet.position.dx - 5,
                        top: bullet.position.dy - 5,
                        child: Container(
                          width: 10,
                          height: 10,
                          decoration: const BoxDecoration(
                            color: Colors.blue,
                            shape: BoxShape.circle,
                          ),
                        ),
                      )),
                  Positioned(
                    left: cursorPosition.dx - 25,
                    top: cursorPosition.dy - 25,
                    child: Transform.rotate(
                      angle: cursorAngle,
                      child: CustomPaint(
                        size: const Size(50, 50),
                        painter: CursorPainter(),
                      ),
                    ),
                  ),
                  Positioned(
                    top: 10,
                    left: 10,
                    child: Text(
                      "Timer: $timerText",
                      style: const TextStyle(color: Colors.white, fontSize: 20),
                    ),
                  ),
                ],
              ),
            ),
        ],
      ),
    );
  }

  String _formatDuration(Duration duration, {bool isGameOver = false}) {
    final minutes = duration.inMinutes;
    final seconds = duration.inSeconds % 60;
    if (isGameOver) {
      return 'You lasted ${minutes.toString()} minutes and ${seconds.toString()} seconds';
    }
    return '${minutes.toString().padLeft(2, '0')}:${seconds.toString().padLeft(2, '0')}';
  }
}

class CursorPainter extends CustomPainter {
  @override
  void paint(Canvas canvas, Size size) {
    final paint = Paint()
      ..color = Colors.white
      ..style = PaintingStyle.stroke
      ..strokeWidth = 2;

    final path = Path();

    path.moveTo(size.width, size.height * 0.5);

    path.lineTo(0, size.height * 0.7);

    path.lineTo(size.width * 0.2, size.height * 0.5);

    path.lineTo(0, size.height * 0.3);

    path.close();

    canvas.drawPath(path, paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}
