import 'dart:ui';
import 'dart:math';

class Particle {
  Offset position;
  final double size;
  Offset velocity;

  Particle({
    required this.position,
    required this.size,
    required this.velocity,
  });
}

class Bullet {
  Offset position;
  final Offset velocity;

  Bullet({
    required this.position,
    required this.velocity,
  });
}

class ParticleModel {
  final List<Particle> _particles = [];
  final List<Bullet> _bullets = [];
  final double _minParticleSize = 25.0;
  final double _maxParticleSize = 80.0;
  final int _numberOfParticles = 15;
  final double _minSpeed = 0.5;
  final double _maxSpeed = 2.0;

  List<Particle> get particles => List.unmodifiable(_particles);
  List<Bullet> get bullets => List.unmodifiable(_bullets);

  void generateParticles(Size screenSize) {
    final random = Random();
    _particles.clear();

    for (int i = 0; i < _numberOfParticles; i++) {
      double size = _minParticleSize +
          random.nextDouble() * (_maxParticleSize - _minParticleSize);
      Offset position = Offset(
        random.nextDouble() * (screenSize.width - size),
        random.nextDouble() * (screenSize.height - size),
      );
      double velocityX = _minSpeed +
          random.nextDouble() *
              (_maxSpeed - _minSpeed) *
              (random.nextBool() ? 1 : -1);
      double velocityY = _minSpeed +
          random.nextDouble() *
              (_maxSpeed - _minSpeed) *
              (random.nextBool() ? 1 : -1); 

      _particles.add(Particle(
        position: position,
        size: size,
        velocity: Offset(velocityX, velocityY),
      ));
    }
  }

  void shootBullet(Offset startPosition, Offset direction) {
    final bulletVelocity = direction * 5.0; // Adjust speed as needed
    _bullets.add(Bullet(
      position: startPosition,
      velocity: bulletVelocity,
    ));
  }

  void updateParticlePositions(Size screenSize) {
    for (var particle in _particles) {
      Offset newPosition = particle.position + particle.velocity;

      
      if (newPosition.dx <= 0 || newPosition.dx + particle.size >= screenSize.width) {
        particle.velocity = Offset(-particle.velocity.dx, particle.velocity.dy);
      }
      if (newPosition.dy <= 0 || newPosition.dy + particle.size >= screenSize.height) {
        particle.velocity = Offset(particle.velocity.dx, -particle.velocity.dy);
      }

      // Update the position
      particle.position += particle.velocity;
    }

    updateBulletPositions(screenSize);
  }

  void updateBulletPositions(Size screenSize) {
    _bullets.removeWhere((bullet) => !screenSize.contains(bullet.position));
    for (var bullet in _bullets) {
      bullet.position += bullet.velocity;
    }
    _checkBulletCollisions();
  }

  void _checkBulletCollisions() {
    _bullets.removeWhere((bullet) {
      for (var particle in _particles) {
        double dx = (particle.position.dx + particle.size / 2) - bullet.position.dx;
        double dy = (particle.position.dy + particle.size / 2) - bullet.position.dy;
        double distance = sqrt(dx * dx + dy * dy);

        if (distance <= (particle.size / 2 + 5)) { // 5 is the radius of the bullet
          _particles.remove(particle);
          return true; // Remove bullet upon collision
        }
      }
      return false;
    });
  }
}
