import 'dart:async';
import 'dart:math';

import 'package:asteroids/model/particle_model.dart';
import 'package:asteroids/view/asteroids_view.dart';
import 'package:flutter/material.dart';

class AsteroidsController extends StatefulWidget {
  const AsteroidsController({super.key});

  @override
  AsteroidsControllerState createState() => AsteroidsControllerState();
}

class AsteroidsControllerState extends State<AsteroidsController>
    with SingleTickerProviderStateMixin {
  late ParticleModel _particleModel;
  late AnimationController _controller;
  late Timer _timer;
  bool _isStopped = false;
  Offset _cursorPosition = const Offset(0, 0);
  Offset _previousCursorPosition =
      const Offset(0, 0); // Track previous position
  Duration _elapsedTime = Duration.zero;

  @override
  void initState() {
    super.initState();
    _particleModel = ParticleModel();
    _initializeAnimation();
  }

  void _initializeAnimation() {
    WidgetsBinding.instance.addPostFrameCallback((_) {
      _generateParticles();
    });
    _controller = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 16),
    )..addListener(() {
        if (!_isStopped) {
          _updateParticlePositions();
          _checkCollision();
        }
      });

    _controller.repeat();

    _timer = Timer.periodic(const Duration(seconds: 1), (timer) {
      if (!_isStopped) {
        setState(() {
          _elapsedTime += const Duration(seconds: 1);
        });
      }
    });
  }

  void _generateParticles() {
    final screenSize = MediaQuery.of(context).size;
    _particleModel.generateParticles(screenSize);
  }

  void _updateParticlePositions() {
    final screenSize = MediaQuery.of(context).size;
    setState(() {
      _particleModel.updateParticlePositions(screenSize);
    });
  }

  void _checkCollision() {
    const cursorRadius = 25.0; // Radius of the cursor icon

    for (var particle in _particleModel.particles) {
      double dx =
          (particle.position.dx + particle.size / 2) - _cursorPosition.dx;
      double dy =
          (particle.position.dy + particle.size / 2) - _cursorPosition.dy;
      double distance = sqrt(dx * dx + dy * dy);

      if (distance <= (particle.size / 2 + cursorRadius)) {
        _stopAnimation();
        break;
      }
    }
  }

  void _stopAnimation() {
    setState(() {
      _isStopped = true;
    });
    _controller.stop();
    _timer.cancel(); // Stop the timer
  }

  void _restartAnimation() {
    setState(() {
      _isStopped = false;
      _elapsedTime = Duration.zero; // Reset elapsed time
    });
    _generateParticles();
    _controller.repeat();
    _timer = Timer.periodic(const Duration(seconds: 1), (timer) {
      if (!_isStopped) {
        setState(() {
          _elapsedTime += const Duration(seconds: 1);
        });
      }
    });
  }

  @override
  Widget build(BuildContext context) {
    final dx = _cursorPosition.dx - _previousCursorPosition.dx;
    final dy = _cursorPosition.dy - _previousCursorPosition.dy;
    final cursorAngle = atan2(dy, dx);
    final direction = Offset(cos(cursorAngle), sin(cursorAngle)); // Direction of bullet

    return AsteroidsView(
      particles: _particleModel.particles,
      bullets: _particleModel.bullets,
      cursorPosition: _cursorPosition,
      cursorAngle: cursorAngle, 
      isStopped: _isStopped,
      elapsedTime: _elapsedTime,
      onMouseHover: (position) {
        if (!_isStopped) {
          setState(() {
            _previousCursorPosition =
                _cursorPosition; // Update previous position
            _cursorPosition = position; // Update current position
          });
        }
      },
      onMouseClick: () {
        if (!_isStopped) {
          _particleModel.shootBullet(_cursorPosition, direction);
        }
      },
      onTryAgainPressed: _restartAnimation,
    );
  }

  @override
  void dispose() {
    _controller.dispose();
    _timer.cancel();
    super.dispose();
  }
}
